﻿# 获取脚本参数（如果有）
[CmdletBinding()]
param(
	[parameter(Mandatory = $false, HelpMessage = "静默")]
	[Switch]$Force,
	[string]$secure_password
)

# 压缩包名称
$CPname = "lang-php-latest-master"

# 签名 GPG KEY-ID
$GpgKI = "2499B7924675A12B"

# 压缩包保存到
$CPSaveTo = "$PSScriptRoot\Release"

# 压缩包临时目录
$TempFolder = "$env:userprofile\Downloads\Temp.$CPname"

# 生成压缩包格式
# 生成 gz, xz，需生成 tar，否则无法创建。
$BuildType = @(
	[ZipFormat]::zip
	[ZipFormat]::tar
	[ZipFormat]::gz
	[ZipFormat]::xz
)

# 压缩包排除文件或目录
$ArchiveExclude = @(
	"-xr-!Release"
	"-xr-!themes\docs"
	"-xr-!themes\downloads"
)

Enum ZipFormat
{
	zip
	tar
	xz
	gz
}

function Set-Password {
	Write-Host "`n  请输入 PGP 安全密码!`n" -ForegroundColor Green
	$script:secure_password = read-host "  证书密码"
	Write-Host ""
}

function Wait-Exit {
	param(
		[int]$wait
	)
	Write-Host "`n   提示：$wait 秒后自动退出该脚本..." -ForegroundColor Red
	Start-Sleep -s $wait
	exit
}

function Get-Zip {
	if (Test-Path "$env:ProgramFiles\7-Zip\7z.exe") {
		$script:Zip = "$env:ProgramFiles\7-Zip\7z.exe"
		return
	}

	if (Test-Path "$env:ProgramFiles(x86)\7-Zip\7z.exe") {
		$script:Zip = "$env:ProgramFiles(x86)\7-Zip\7z.exe"
		return
	}

	if (Test-Path "$env:SystemDrive\Yi\Yi\AIO\7z.exe") {
		$script:Zip = "$env:SystemDrive\Yi\Yi\AIO\7z.exe"
		return
	}
	$script:Zip = "No"
}

function Clear-old-file {
	remove-item -path "$CPSaveTo" -Recurse -force -ErrorAction SilentlyContinue
	remove-item -path "$TempFolder" -Recurse -force -ErrorAction SilentlyContinue
}

function Unpack-packer {
	foreach ($list in $BuildType ) {
		Unpack-packer-Create -Type $list
	}
}

function Unpack-packer-Create {
	Param(
		[string]$Type
	)

	Check-SD -chkpath $TempFolder

	switch ($Type) {
		"zip" {
			Write-Host "  * 正在生成 $CPname.zip"
			$arguments = "a", "-tzip", "$TempFolder\$CPname.zip", "$ArchiveExclude", "*.*", "-mcu=on", "-r", "-mx9";
		}
		"tar" {
			Write-Host "  * 正在生成 $CPname.tar"
			$arguments = "a", "$TempFolder\$CPname.tar", "$ArchiveExclude", "*.*", "-r";
		}
		"xz" {
			Write-Host "  * 正在生成 $CPname.tar.xz"
			if (Test-Path "$TempFolder\$CPname.tar") {                
				$arguments = "a", "$TempFolder\$CPname.tar.xz", "$TempFolder\$CPname.tar", "-mf=bcj", "-mx9";
			} else {
				Write-Host "    - 跳过生成，未找到 $CPname.tar`n"
				return
			}
		}
		"gz" {
			Write-Host "  * 正在生成 $CPname.tar.gz"
			if (Test-Path "$TempFolder\$CPname.tar") {           
				$arguments = "a", "-tgzip", "$TempFolder\$CPname.tar.gz", "$TempFolder\$CPname.tar", "-mx9";
			} else {
				Write-Host "  x 跳过生成，未找到 $CPname.tar`n"
				return
			}
		}
	}
	Start-Process $script:Zip "$arguments" -Wait -WindowStyle Minimized
	Write-Host "    - 完成`n" -ForegroundColor Green
}

function Get-SCFile {
	param($opath,$shortname,$Report)
	$fullnewpath = $opath + "." + $Report

	# Add the direct generation command to determine whether it is ASC or SHA256, and add judgment.
	Switch ($Report)
	{
		"asc" {
			Write-Host "  * 正在生成 $CPname.asc"

			Remove-Item -path "${opath}.sig" -Force -ErrorAction SilentlyContinue
			Remove-Item -path "${opath}.asc" -Force -ErrorAction SilentlyContinue
			if ($script:secure_password) {
				Start-Process gpg -argument "--pinentry-mode loopback --passphrase $script:secure_password --local-user $GpgKI --output $opath.asc --detach-sign $opath" -Wait -WindowStyle Minimized
			} else {
				Start-Process gpg -argument "--local-user $GpgKI --output $opath.asc --detach-sign $opath" -Wait -WindowStyle Minimized
			}
			Write-Host "    - 完成`n" -ForegroundColor Green
		}
		"sha256" {
			Write-Host "  * 正在生成 $CPname.sha256"
			$calchash = (Get-FileHash $opath -Algorithm SHA256)
			Remove-Item -path $fullnewpath -Force -ErrorAction SilentlyContinue
			$calchash.hash + "  " + $shortname | Out-File -FilePath $fullnewpath -Encoding ASCII
			Write-Host "    - 完成`n" -ForegroundColor Green
		}
	}
}

function Create-allfileAS {
	remove-item -path "$TempFolder\*.tar" -force -ErrorAction SilentlyContinue

	Get-ChildItem $TempFolder -Recurse -ErrorAction SilentlyContinue | Foreach-Object {
		Get-SCFile -opath $_.FullName -shortname $_.Name -Report "asc"
		Get-SCFile -opath $_.FullName -shortname $_.Name -Report "sha256"
	}
}

function Check-SD {
	Param(
		[string]$chkpath
	)

	if(!(Test-Path $chkpath -PathType Container)) {
		New-Item -Path $chkpath -ItemType Directory -ErrorAction SilentlyContinue | Out-Null
		if(!(Test-Path $chkpath -PathType Container)) {
			Write-Host "    - 创建目录失败：$($chkpath)`n" -ForegroundColor Red
			return
		}
	}
}

function Move-allfile {
	Check-SD -chkpath $CPSaveTo

	Get-ChildItem $TempFolder -Recurse -ErrorAction SilentlyContinue | Foreach-Object {
		Move-Item -Path $_.FullName -Destination $CPSaveTo -ErrorAction SilentlyContinue | Out-Null
	}
	remove-item -path "$TempFolder" -Recurse -force -ErrorAction SilentlyContinue
}

function Get-Mainpage {
	clear
	Write-host "
  Author: Yi ( https://fengyi.tel )

  From: Yi's Solution
  buildstring: 5.1.1.0.bk_release.210120-1208

  任务计划：

     1. 删除旧文件
     2. 打包所有文件（不含排除项）
     3. 生成 $CPname ( .zip, tar.gz, tar.xz )
     4. 生成 ASC 签名文件
     5. 生成 sha256`n"

	If ($Force) {
		Clear-old-file
		Unpack-packer
		Create-allfileAS
		Move-allfile
	} else {
		Set-Password
		Clear-old-file
		Unpack-packer
		Create-allfileAS
		Move-allfile
		Wait-Exit -wait 6
	}
}

Get-Zip
Get-Mainpage